/* --------------------------------------------------------------
 gambio_store_controller 2019-02-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Gambio Store Controller
 *
 * This controller will handles the frontend logic of the Gambio Store page.
 *
 * @module Controllers/gambio_store
 */
gx.controllers.module(
	'gambio_store',
	['xhr', `${gx.source}/libs/auto_updater`],
	function({storeUrl, storeToken}) {
		/**
		 * Module element
		 *
		 * @type {HTMLElement}
		 */
		const parent = this[0];
		
		/**
		 * Iframe element
		 *
		 * @type {HTMLIFrameElement}
		 */
		let iframe;
		
		/**
		 * Return shop information
		 *
		 * @returns {Promise<Object>}
		 */
		function fetchShopInfo() {
			function handle(resolve) {
				jse.libs.xhr
					.get({url: './admin.php?do=GambioStoreAjax/collectShopInformation'})
					.done(resolve);
			}
			
			return new Promise(handle)
		}
		
		/**
		 * Return whether the data processing has been accepted.
		 *
		 * @returns {Promise<Object>}
		 */
		function isDataProcessingAccepted() {
			function handle(resolve) {
				jse.libs.xhr
					.get({url: './admin.php?do=GambioStoreAjax/isDataProcessingAccepted'})
					.done(resolve);
			}
			
			return new Promise(handle)
		}
		
		/**
		 * Build Iframe element and resolve returned promise once the content is loaded
		 *
		 * @param {string} url Iframe URL
		 * @returns {Promise<Event>}
		 */
		function buildIFrame(url) {
			/**
			 * Build element
			 *
			 * @param {function} resolve
			 */
			function handle(resolve) {
				iframe = document.createElement('iframe');
				
				const language = jse.core.config.get('languageCode');
				storeUrl += storeUrl.includes('?') ? '&language=' + language : '?language=' + language;
				
				iframe.src = storeUrl;
				iframe.style.width = '100%';
				iframe.style.height = '100%';
				iframe.style.border = 'none';
				iframe.style.overflow = 'hidden';
				
				iframe.addEventListener('load', resolve);
				
				parent.appendChild(iframe);
			}
			
			return new Promise(handle);
		}
		
		/**
		 * Attach message event listener
		 */
		function listenToMessages() {
			/**
			 * Handle message event
			 *
			 * @param {MessageEvent} event
			 */
			function onMessage({data}) {
				const {type, payload} = data;
				
				if (type === 'response_iframe_height') {
					iframe.style.height = `${payload.height}px`;
					return;
				}
				
				if (type === 'updates_counter') {
					setUpdatesCounter(payload.updatesCounter);
					return;
				}
				
				if (type === 'update_shop_information' || type === 'request_shop_information') {
					fetchShopInfo().then(sendShopInfo);
					return;
				}
				
				if (type === 'request_store_token') {
					sendMessage('send_store_token', {'storeToken': storeToken});
					return;
				}
				
				if (type === 'send_data_processing_accepted') {
					window.location.href = 'admin.php?do=GambioStore/AcceptDataProcessing';
					return;
				}
				
				if (type === 'start_installation_process') {
					install(payload);
					return;
				}
				
				if (type === 'uninstall_theme') {
					uninstallTheme(payload);
				}
				
				if (type === 'reload_page') {
					window.location.reload(true);
				}
				
				if (type === 'activate_theme') {
					activateTheme(payload);
				}
			}
			
			window.addEventListener('message', onMessage);
		}
		
		/**
		 * Activate a theme.
		 *
		 * @param payload
		 */
		function activateTheme(payload) {
			jse.libs.xhr
				.post({
					url: './admin.php?do=GambioStore/ActivateTheme',
					data: {'themeStorageName': payload.themeStorageName}
				}).done(() => {
				sendMessage('activation_succeeded');
			}).fail(() => {
				sendMessage('activation_failed');
			});
		}
		
		/**
		 * Send message to Iframe
		 *
		 * @param {string} type Message type
		 * @param {object} [payload] Message payload
		 */
		function sendMessage(type, payload) {
			iframe.contentWindow.postMessage({type, payload}, '*');
		}
		
		/**
		 * Send shop information data to Iframe
		 *
		 * @param {Object} shopInfo Shop information data
		 */
		function sendShopInfo(shopInfo) {
			sendMessage('send_shop_information', {shopInfo})
		}
		
		/**
		 * Install a package
		 *
		 * @param {Object} data Package data
		 */
		function install(data) {
			const $installingPackageModal = $('.installing-package.modal');
			const $progressDescription = $installingPackageModal.find('.progress-description');
			const $progressBar = $installingPackageModal.find('.progress .progress-bar');
			
			$progressDescription.text(jse.core.lang.translate('PREPARING_PACKAGE', 'gambio_store'));
			
			const updateProgressCallback = ({progress}) => {
				let progressPercentage = Math.ceil(progress * 100);
				
				if (progressPercentage < 0) {
					progressPercentage = 0;
				} else if (progressPercentage > 100) {
					progressPercentage = 100;
				}
				
				$progressBar.prop('aria-valuenow', progressPercentage);
				$progressBar.css('width', progressPercentage + '%');
				$progressBar.text(progressPercentage + '%');
			};
			
			updateProgressCallback({progress: 0}); // reset
			
			$installingPackageModal.modal('show');
			
			jse.libs.auto_updater
				.checkGambioStorePackageFilePermissions(data)
				.then(() => {
					$progressDescription.text(jse.core.lang.translate('INSTALLING_PACKAGE', 'gambio_store'));
					
					jse.libs.auto_updater
						.installGambioStorePackage(data, updateProgressCallback)
						.then(() => {
							
							// By checking whether a gallery object is present,
							// we can determine if this is a theme or not.
							if (data.details.gallery) {
								// find out if the installed theme was an update of the currently active theme
								jse.libs.xhr.get({
									url: './admin.php?do=GambioStoreAjax/IsThemeActive&themeName='
										+ data.details.fileName
								}).done(response => {
										if (response.isActive === true) {
											// and if so, activate it again for the content manager entries
											jse.libs.xhr
												.post({
													url: './admin.php?do=GambioStore/ActivateTheme',
													data: {'themeStorageName': data.details.fileName}
												}).done(() => {
												sendMessage('installation_succeeded')
											}).fail(() => {
												sendMessage('installation_failed')
											})
										} else {
											sendMessage('installation_succeeded')
										}
									}).fail(() => {
										sendMessage('installation_failed')
									});
								} else {
									sendMessage('installation_succeeded')
								}
							})
							.catch(() =>
								sendMessage('installation_failed')
							)
							.finally(() => {
								updateProgressCallback({progress: 1});
								setTimeout(() => {
									$installingPackageModal.modal('hide');
								}, 2000);
							});
					}
				)
				.catch(() => sendMessage('ftp_data_requested'));
		}
		
		/**
		 * Uninstall a theme
		 *
		 * @param {Object} data Theme data
		 */
		function uninstallTheme(data) {
			jse.libs.auto_updater
				.uninstallTheme(data.filename)
				.then(() => sendMessage('uninstall_succeeded'))
				.catch((data) => sendMessage('uninstall_failed', data));
		}
		
		/**
		 * Applies a darker background color to the page.
		 */
		function adjustBackgroundColor() {
			document.getElementById('main-content').style.background = '#f5f5f5';
		}
		
		/**
		 * Set the updates counter badge in the page navigation row.
		 *
		 * @param {Number} updatesAvailable
		 */
		function setUpdatesCounter(updatesAvailable) {
			const className = 'gambio-store-updates-counter';
			
			let updatesCounter = document.getElementsByClassName(className);
			
			if (updatesAvailable === 0) {
				if (updatesCounter.length && updatesCounter.style) {
					updatesCounter.style.display = "none";
				}
				return;
			} else {
				if (updatesCounter.length && updatesCounter.style) {
					updatesCounter.style.display = "inline";
				}
			}
			
			if (updatesCounter.length) {
				updatesCounter.innerHTML = updatesAvailable;
				return;
			}
			
			updatesCounter = document.createElement('span');
			updatesCounter.className = 'gambio-store-updates-counter';
			updatesCounter.innerHTML = updatesAvailable;
			
			const navItem = document.querySelector('#main-content .content-navigation .nav-item:last-child');
			
			if (!navItem) {
				return;
			}
			
			if (navItem.classList.contains('no-link')) {
				navItem.appendChild(updatesCounter);
			} else {
				navItem.querySelector('a').appendChild(updatesCounter);
			}
		}
		
		/**
		 * Initialize
		 *
		 * @param {Function} done JSEngine callback function for ready state indication
		 */
		function init(done) {
			adjustBackgroundColor();
			
			function proceed({accepted}) {
				let url = storeUrl + (accepted ? '' : 'dataprocessing')
				
				listenToMessages();
				
				return buildIFrame(url);
			}
			
			isDataProcessingAccepted()
				.then(proceed)
				.then(done);
		}
		
		return {init};
	}
);
